/*
 * Lachesis, an IRCRPG combat engine - upper-level IRC Interface
 * Copyright 2002 M. Dennis
 *
 * This program is free software; please see the file COPYING for details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/utsname.h>
#include "lachesis.h"
#include "irc.h"
#include "ircint.h"
#include "auth.h"
#include "utils.h"
#include "types.h"
#include "log.h"

extern CmdOpts options;
#ifndef NO_USERID
extern char *username;
#endif

char *irc_curchan;
IRC_PCL irc_channels = NULL;

TF irc_interface_active = FALSE;

#ifndef DFL_IRCNICK
#define DFL_IRCNICK "Lachesis"
#endif

#ifndef DFL_IRCNAME
#define DFL_IRCNAME "Measuring the threads of Fate"
#endif

#ifndef DFL_IRCUSER
#define DFL_IRCUSER "lachesis"
#endif

void ParseRPL_USERHOST(const char *rpl, char *nick, char *user, char *host,
		       uint8 *flags)
{
  int olen;
  const char *t1, *t2, *str = rpl;
  if (*str==':')
    str++;
  if (flags!=NULL)
    *flags = 0;
  t1 = strchr(str, '=');
  olen = t1 - str;
  if (*(t1 - 1) == '*') {
    olen--;
    if (flags!=NULL)
      *flags += UHFLAG_OPER;
  }
  if (olen>=ALLOC_NICK) {
    fprintf(stderr, "Error: Nick in RPL_USERHOST exceeds ALLOC_NICK-1: %s\n",
	    str);
    exit (255);
  }
  if (nick!=NULL)
    Util_Copy(nick, str, olen+1);
  t1++;
  if (flags!=NULL)
    if (*t1 == '+')
      *flags += UHFLAG_AWAY;
  t1++;
  t2 = strchr(t1, '@');
  olen = t2 - t1;
  if (olen>=ALLOC_USER) {
    fprintf(stderr, "Error: User in RPL_USERHOST exceeds ALLOC_USER-1: %s\n",
	    str);
    exit (255);
  }
  if (user!=NULL)
    Util_Copy(user, t1, olen+1);
  if (host!=NULL)
    Util_Copy(host, t2+1, ALLOC_HOST);
}

void IntConnectionReady()
{
  Auth_Connect();
  char *key;
  if (options.key[0])
    key = options.key;
  else
    key = NULL;
  if (options.channel[0]) {
    if (options.channel[0]=='#'||options.channel[0]=='&'||
	options.channel[0]=='+')
      IRC_Join(options.channel, key);
    else {
      char *temp;
      temp = new char[ALLOC_CHAN];
      temp[0] = '#';
      Util_Copy(temp+1, options.channel, ALLOC_CHAN-1);
      IRC_Join(temp, key);
      delete [] temp;
    }
#ifdef DFL_IRCCHAN
  } else
    IRC_Join(DFL_IRCCHAN, NULL);
#endif
}

extern "C" 
{
  void ConnectionReady() 
  { 
#ifdef DEBUG
    fputs("DEBUG - ConnectionReady called\n", stderr);
#endif
    IntConnectionReady(); 
  }

  void HandleDCCChatRequest(void *cookie, const char *nick)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleDCCChatRequest called\n", stderr);
#endif
    Auth_DCCRequest(nick, cookie);
  }

  void HandleDCCChatRequestPfx(void *cookie, const char *prefix)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleDCCChatRequestPfx called\n", stderr);
#endif
    Auth_DCCRequestPfx(prefix, cookie);
  }

  void HandleDCCChatOpened(void *cookie)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleDCCChatOpened called\n", stderr);
#endif
    Auth_DCCEstablished(cookie);
  }

  void HandleDCCChatClosed(void *cookie)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleDCCChatClosed called\n", stderr);
#endif
    Auth_DCCLost(cookie);
  }

  void HandleDCCChatMessage(void *cookie, const char *message)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleDCCChatMessage called\n", stderr);
#endif
    Auth_DCCMessage(cookie, message);
  }

  void HandleNumeric(const int numeric, const char *message)
  {
    const char *temp;
#ifdef DEBUG
    fputs("DEBUG - HandleNumeric called\n", stderr);
#endif
    // For some reason these messages get padded...
    temp = message;
    while (*temp == 32)
      temp++;
    switch(numeric) {
    case 302:
      char nick[ALLOC_NICK], user[ALLOC_USER], host[ALLOC_HOST];
#ifdef DEBUG
      fputs("DEBUG - Numeric 302 handler\n", stderr);
#endif
      ParseRPL_USERHOST(temp, nick, user, host, NULL);
#ifdef DEBUG
      fprintf(stderr, "DEBUG - ParseRPL_USERHOST %s %s %s\n", nick, user, host);
#endif
      Auth_Userhost(nick, user, host);
      break;
    }
  }

  void HandleUserhost(const char *nick, const char *user, const char *host)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleUserhost called\n", stderr);
#endif
    // Depending on the sophistication of the interface, it may forward
    // USERHOST-requests here, or simply forward 302s as above.
    Auth_Userhost(nick, user, host);
  }

  void HandleNotice(const char *prefix, const char *target, const char *message)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleNotice called\n", stderr);
#endif
    if (strcasecmp(IRC_HostmaskNick(prefix), "NICKSERV")==0) {
      if (strcasecmp(IRC_HostmaskUser(prefix), "SERVICES")==0) {
	if (strcasecmp(target, IRC_GetNick())==0)
	  Auth_NotifyMaster(Util_Format("-%s- %s", prefix, message, NULL));
	else
	  Auth_NotifyMaster(Util_Format("-%s:%s- %s", prefix, target, message));
      }
    }
  }

  const char * HandleCTCP(const char *prefix, const char *target,
			  const char *message)
  {
    const char *data;
    const char *command;

#ifdef DEBUG
    fputs("DEBUG - HandleCTCP called\n", stderr);
#endif
    data = message;
    command = Util_Separate(&data, 32);
    if (command==NULL) {
      command = message;
      data = NULL;
    }

    if (strcasecmp(command, "CLIENTINFO")==0) {
      if (data!=NULL&&*data) {
	if (strcasecmp(data, "ACTION")==0) {
	  return "ACTION contains action descriptions for atmosphere";
	}
	if (strcasecmp(data, "DCC")==0) {
	  return "DCC requests a direct_client_connection";
	}
	if (strcasecmp(data, "VERSION")==0) {
	  return "VERSION shows client type, version, and environment";
	}
	if (strcasecmp(data, "CLIENTINFO")==0) {
	  return "CLIENTINFO gives information about available CTCP commands";
	}
	if (strcasecmp(data, "USERINFO")==0) {
	  return "USERINFO returns user settable information";
	}
	if (strcasecmp(data, "ERRMSG")==0) {
	  return "ERRMSG returns error messages";
	}
	if (strcasecmp(data, "FINGER")==0) {
	  return "FINGER is treated as synonymous with USERINFO by this client";
	}
	if (strcasecmp(data, "PING")==0||strcasecmp(data, "ECHO")==0) {
	  return Util_Format("%s returns the arguments it receives", data, NULL, NULL);
	}
	return Util_Format("%s is not a valid function", data, NULL, NULL);
      }
      return "ACTION DCC VERSION CLIENTINFO USERINFO ERRMSG FINGER PING ECHO  :Use CLIENTINFO <COMMAND> to get more specific information";
    }
    if (strcasecmp(command, "VERSION")==0) {
      struct utsname our_uname;
      char nick[ALLOC_NICK], buf[ALLOC_BUFFER];
      uname(&our_uname);
      Util_Copy(nick, IRC_HostmaskNick(prefix), ALLOC_NICK);
      snprintf(buf, ALLOC_BUFFER, "Lachesis IRCRPG combat engine %s on %s %s",
	       Lachesis_Version(), our_uname.sysname, our_uname.release);
      IRC_CTCPReply(nick, "VERSION", buf);
      IRC_CTCPReply(nick, "VERSION", IRC_VersionInfo());
      return NULL;
    }
    if (strcasecmp(command, "USERINFO")==0||strcasecmp(command, "FINGER")==0) {
      if (options.userinfo[0])
	return options.userinfo;
      return "There is no USERINFO set.";
    }
    if (strcasecmp(command, "PING")==0||strcasecmp(command, "ECHO")==0) {
      IRC_CTCPReply(IRC_HostmaskNick(prefix), command, data);
      return NULL;
    }

    // Note: If planning to handle actions, they are sent here.
    if (strcasecmp(command, "ACTION")==0)
      if (strcasecmp(target, IRC_GetNick())!=0 &&
	  (strcasecmp(target, irc_curchan)==0 ||
	   options.logother))
	Log_Action(IRC_HostmaskNick(prefix), data);
    return NULL;
  }

  void HandleCTCPReply(const char *prefix, const char *message)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleCTCPReply called\n", stderr);
#endif
    // TODO: handle any legitimate CTCP replies
  }

  void HandleNames(const char *channel, const char **names, int count)
  {
    int c;
#ifdef DEBUG
    fputs("DEBUG - HandleNames called\n", stderr);
#endif

    if (!HLIRC_CheckChan(channel))
      return ;
    for (c=0;c<count;c++) {
      if (names[c][0] == '@' || names[c][0] == '+')
	Auth_Track(names[c]+1, channel, TRUE);
      else
	Auth_Track(names[c], channel, TRUE);
    }
  }

  void HandleName(const char *channel, const char *name)
  {
#ifdef DEBUG
    fprintf(stderr, "DEBUG - HandleName called: %s %s\n", channel, name);
#endif
    if (!HLIRC_CheckChan(channel))
      return ;
    if (name[0] == '@' || name[0] == '+') {
      if (strcasecmp(name+1, IRC_GetNick())!=0)
	Auth_Track(name+1, channel, TRUE);
    } else {
      if (strcasecmp(name, IRC_GetNick())!=0)
	Auth_Track(name, channel, TRUE);
    }
  }

  void HandleInvite(const char *prefix, const char *channel)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleInvite called\n", stderr);
#endif
    // TODO: Tie into authentication system
  }

  void HandleKick(const char *nick, const char *target,
		  const char *channel, const char *message)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleKick called\n", stderr);
#endif
    if (!HLIRC_CheckChan(channel))  // This shouldn't happen
      return ;
    if (strcasecmp(nick, IRC_GetNick())==0)
      HLIRC_RemChan(channel);
    else
      Auth_Track(nick, channel, FALSE);
  }

  void HandleMSG(const char *prefix, const char *message)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleMSG called\n", stderr);
#endif
    Auth_IRCMessage(prefix, message);
  }

  void HandlePublic(const char *prefix, const char *target,
		    const char *message)
  {
#ifdef DEBUG
    fputs("DEBUG - HandlePublic called\n", stderr);
#endif
    // TODO: Whatever may be necessary
    if (strcasecmp(target, irc_curchan)==0 || options.logother) {
      Auth_PUser user = Auth_TUser::MatchPrefix(prefix);
      Log_Public(user, IRC_HostmaskNick(prefix), message);
    }
  }

  void HandleJoined(const char *channel)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleJoined called\n", stderr);
#endif
    Util_Copy(irc_curchan, channel, ALLOC_CHAN);
    HLIRC_AddChan(channel);
  }

  void HandleJoin(const char *prefix, const char *channel)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleJoin called\n", stderr);
#endif
    Auth_TrackPr(prefix, channel);
  }

  void HandleOurNickChanged(const char *nick)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleOurNickChanged called\n", stderr);
#endif
    // TODO: Tie into authentication system if applicable?
  }

  void HandleNickChange(const char *prefix, const char *nick)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleNickChange called\n", stderr);
#endif
    Auth_Nick(prefix, nick);
  }

  void HandlePart(const char *nick, const char *channel, const char *message)
  {
#ifdef DEBUG
    fputs("DEBUG - HandlePart called\n", stderr);
#endif
    if (strcasecmp(nick, IRC_GetNick())==0)
      HLIRC_RemChan(channel);
    else
      Auth_Track(nick, channel, 0);
  }

  void HandleQuit(const char *nick, const char *message)
  {
#ifdef DEBUG
    fputs("DEBUG - HandleQuit called\n", stderr);
#endif
    Auth_Track(nick, NULL, FALSE);
  }

  void HandleDisconnect(const char *message) {
#ifdef DEBUG
    fputs("DEBUG - HandleDisconnect called\n", stderr);
#endif
    Auth_Disconnect(message);
    HLIRC_CleanChan();
    free(irc_curchan);
    irc_curchan = NULL;
#ifdef AUTO_RECONNECT
    if (irc_interface_active)
      HLIRC_BeginIRCInterface();
#else
    irc_interface_active = FALSE;
    Shutdown();
#endif
  }

}

void HLIRC_BeginIRCInterface()
{
  char *nick, *name, *user, *server;
  short int port;
#ifdef DEBUG
  fputs("DEBUG - HLIRC_BeginIRCInterface called\n", stderr);
#endif
  Log_Init();
  if (irc_curchan==NULL) {
    irc_curchan = (char *)(malloc(ALLOC_CHAN));
    *irc_curchan = 0;
  }
  if (*options.ircnick)
    nick = options.ircnick;
  else
    nick = DFL_IRCNICK;
  if (*options.ircname)
    name = options.ircname;
  else
    name = DFL_IRCNAME;
  if (*options.ircuser)
    user = options.ircuser;
  else
#ifndef NO_USERID
    user = username;
#else
    user = DFL_IRCUSER;
#endif

  server = options.server;
  if (options.port)
    port = options.port;
  else
    port = 6667;

  irc_interface_active = TRUE;
  IRC_Connect(server, port, nick, user, name);
}

void HLIRC_EndIRCInterface()
{
#ifdef DEBUG
  fputs("DEBUG - HLIRC_EndIRCInterface called\n", stderr);
#endif
  Log_Cleanup();
  irc_interface_active = FALSE;
  if (irc_curchan!=NULL) {
    free(irc_curchan);
    irc_curchan = NULL;
  }
  HLIRC_CleanChan();
  if (IRC_IsConnected()) {
    if (options.quitmessage[0])
      IRC_Quit(options.quitmessage);
    else
      IRC_Quit("IRC Interface Terminated");
  }
}

void HLIRC_AddChan(const char *channel)
{
#ifdef DEBUG
  fputs("DEBUG - HLIRC_AddChan called\n", stderr);
#endif
  irc_channels = new IRC_TCL(irc_channels);
  Util_Copy(irc_channels->chan, channel, ALLOC_CHAN);
}

void HLIRC_RemChan(const char *channel)
{
#ifdef DEBUG
  fputs("DEBUG - HLIRC_RemChan called\n", stderr);
#endif
  if (irc_channels==NULL)
    return ;
  IRC_PCL temp;
  if (strcasecmp(irc_channels->chan, channel)==0) {
    if ((temp = irc_channels->GetNext())!=NULL) {
      IRC_TCL::Delete(irc_channels);
      irc_channels = temp;
      if (strcasecmp(irc_curchan, channel)==0)
	Util_Copy(irc_curchan, temp->chan, ALLOC_CHAN);
      Auth_RemTrack(channel);
    } else {
      IRC_TCL::Delete(irc_channels);
      irc_channels = NULL;
      *irc_curchan = 0;
      Auth_RemTrack(channel);
    }
  }
  temp = irc_channels->GetNext();
  while (temp!=NULL) {
    if (strcasecmp(temp->chan, channel)==0) {
      if (strcasecmp(irc_curchan, channel)==0)
	Util_Copy(irc_curchan, irc_channels->chan, ALLOC_CHAN);
      IRC_TCL::Delete(temp);
      Auth_RemTrack(channel);
      return ;
    }
    temp = temp->GetNext();
  }
}

TF HLIRC_CheckChan(const char *channel)
{
#ifdef DEBUG
  fputs("DEBUG - HLIRC_CheckChan called\n", stderr);
#endif
  if (irc_channels==NULL)
    return FALSE;
  IRC_PCL temp;
  for (temp=irc_channels;temp!=NULL;temp=temp->GetNext())
    if (strcasecmp(temp->chan, channel)==0)
      return TRUE;
  return FALSE;
}

void HLIRC_CleanChan()
{
#ifdef DEBUG
    fputs("DEBUG - HLIRC_CleanChan called\n", stderr);
#endif
  if (irc_channels==NULL)
    return ;
#ifdef DEBUG
    fputs("DEBUG - HLIRC_CleanChan is calling IRC_TCL::Flush\n", stderr);
#endif
  IRC_TCL::Flush(irc_channels);
  irc_channels = NULL;
#ifdef DEBUG
  fputs("DEBUG - HLIRC_CleanChan is deleting irc_curchan\n", stderr);
#endif
  if (irc_curchan!=NULL)
    *irc_curchan = 0;
#ifdef DEBUG
  fputs("DEBUG - HLIRC_CleanChan is calling Auth_FlushTrack\n", stderr);
#endif
  Auth_FlushTrack();
}

void HLIRC_Cleanup()
{
#ifdef DEBUG
    fputs("DEBUG - HLIRC_Cleanup called\n", stderr);
#endif
  if (irc_interface_active)
    HLIRC_EndIRCInterface();
  LowIrcCleanup();
}
