/*
 * Lachesis, an IRCRPG combat engine - Message-based IRC Interface
 * Copyright 2002 M. Dennis
 *
 * This interface launches a standalone "IRC connection service"
 * using BotNet, and communicates with it via a socket.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include "fdio.h"
#include "../lachesis.h"
#include "../types.h"
#include "../irc.h"
#include "../utils.h"
#include "../alloc.h"
#include "../ircint.h"

fd_set *read_fds = NULL;
int icsfd = -1;
char bn2_our_nick[ALLOC_NICK];
char bn2_our_name[ALLOC_NAME];
char bn2_our_user[ALLOC_USER];
char bn2_our_host[ALLOC_HOST];
char bn2_internal_buffer[ALLOC_MINIBUFFER];
char *ICSINFO = NULL;
TF icsint_active = FALSE;
unsigned long bn2_our_addr; // For now, we take this from BotNet.
extern CmdOpts options;

typedef struct tag_dcc_ticket {
  char nick[ALLOC_NICK];
  int sock;
  TF listen;
  TF stale;
  struct sockaddr_in addr;  // To whom we are connected (or will connect)
  socklen_t addrlen;
  struct tag_dcc_ticket *next, *prev;
} TDCC, *PDCC;

PDCC dcc_list = NULL;

void RemDCC(PDCC ticket)
{
  if (ticket == NULL)
    return ;
  if (ticket->sock != -1) {
    FD_CLR(ticket->sock, read_fds);
    close (ticket->sock);
  }
  if (dcc_list == ticket)
    dcc_list = ticket->next;
  if (ticket->next!=NULL)
    ticket->next->prev = ticket->prev;
  if (ticket->prev!=NULL)
    ticket->prev->next = ticket->next;
  free (ticket);
}

void NewDCC(const char *nick, int fd)
{
  PDCC temp;
  temp = (PDCC)malloc(sizeof(TDCC));
  temp->addrlen = sizeof (struct sockaddr_in);
  temp->prev = NULL;
  temp->next = dcc_list;
  if (dcc_list!=NULL)
    dcc_list->prev = temp;
  Util_Copy(temp->nick, nick, ALLOC_NICK);
  dcc_list = temp;
  temp->sock = fd;
  if (fd>-1) {
    FD_SET(fd, read_fds);
    temp->listen = TRUE;
  } else
    temp->listen = FALSE;
  temp->stale = FALSE;
}

void DoDCCCleanup() {
  PDCC t1, t2 = dcc_list;
  while (t2 != NULL) {
    t1 = t2->next;
    if (t2->stale)
      RemDCC(t2);
    t2 = t1;
  }
}

void AcceptDCC(PDCC dcc)
{
  int asock, ioflags;

  if (dcc->sock<0 || !dcc->listen) return ;
  asock = accept(dcc->sock, (struct sockaddr *)&(dcc->addr), 
		 &(dcc->addrlen));
  if (asock<0) {
    fprintf(stderr, "AcceptDCC: Unable to accept connection: %m\n");
    return ;
  }

  //  ioflags=fcntl(asock, F_GETFL, 0);
  //  ioflags|=O_NONBLOCK;
  //  fcntl(asock, F_SETFL, ioflags);

  FD_CLR(dcc->sock, read_fds);
  FD_SET(asock, read_fds);
  close(dcc->sock);
  dcc->sock = asock;
  dcc->listen = FALSE;
  HandleDCCChatOpened((void *)dcc);
}

PDCC GetDCC(const char *nick) // Is this even needed?
{
  PDCC temp = dcc_list;
  while (temp!=NULL) {
    if (strcmp(temp->nick, nick)==0)
      return temp;
    temp = temp->next;
  }
  return NULL;
}

void PrefixSeparate()
{
  char *temp;
  if ((temp=strchr(bn2_internal_buffer, '@')) != NULL)
    *temp = 0;
  if ((temp=strchr(bn2_internal_buffer, '!')) != NULL)
    *temp = 0;
}

void IRC_Userhost(const char *target)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC USERHOST %s\n", target);
#endif
  fdprintf(icsfd, "PQUOTE USERHOST %s\n", target);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Quote(const char *str, TF priority)
{
  const char *qcmd = (priority)?"PQUOTE":"QUOTE";
#ifdef DEBUG
  fprintf(stderr, "DEBUG - Sending to ICS: %s %s\n", qcmd, str);
#endif
  fdprintf(icsfd, "%s %s\n", qcmd, str);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Notice(const char *target, const char *message)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC NOTICE %s %s\n", target, message);
#endif
  fdprintf(icsfd, "QUOTE NOTICE %s :%s\n", target, message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Part(const char *channel, const char *message)
{
  if (message!=NULL&&*message) {
#ifdef DEBUG
    fprintf(stderr, "DEBUG - IRC PART %s :%s\n", channel, message);
#endif
    fdprintf(icsfd, "PQUOTE PART %s :%s\n", channel, message);
  } else {
#ifdef DEBUG
    fprintf(stderr, "DEBUG - IRC PART %s\n", channel);
#endif
    fdprintf(icsfd, "PQUOTE PART %s\n", channel);
  }
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_CTCP(const char *target, const char *command, const char *data)
{
  if (data!=NULL) {
#ifdef DEBUG
    fprintf(stderr, "DEBUG - IRC CTCP %s %s %s\n", target, command, data);
#endif
    fdprintf(icsfd, "MSG %s %c%s %s%c\n", target, 1, command, data, 1);
  } else {
#ifdef DEBUG
    fprintf(stderr, "DEBUG - IRC CTCP %s %s\n", target, command);
#endif
    fdprintf(icsfd, "MSG %s %c%s%c\n", target, 1, command, 1);
  }
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_CTCPReply(const char *target, const char *command, const char *data)
{
  IRC_Notice(target, Util_CTCPFmtD(command, data));
}

const char * IRC_VersionInfo()
{
  return ICSINFO;
}

void IRC_Nick(const char *nick)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC NICK %s\n", nick);
#endif
  fdprintf(icsfd, "NICK %s\n", nick);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Public(const char *channel, const char *message)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC PUBLIC %s %s\n", channel, message);
#endif
  fdprintf(icsfd, "PUBLIC %s %s\n", channel, message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Action(const char *channel, const char *message)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC ACTION %s %s\n", channel, message);
#endif
  fdprintf(icsfd, "ACTION %s %s\n", channel, message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_MSG(const char *target, const char *message)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC MSG %s %s\n", target, message);
#endif
  fdprintf(icsfd, "MSG %s %s\n", target, message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Describe(const char *target, const char *message)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC DESCRIBE %s %s\n", target, message);
#endif
  fdprintf(icsfd, "MSG %s \1ACTION %s\1\n", target, message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Wallop(const char *message)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC WALLOP %s\n", message);
#endif
  fdprintf(icsfd, "WALLOP %s\n", message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Join(const char *channel, const char *key)
{
  if (key!=NULL) {
#ifdef DEBUG
    fprintf(stderr, "DEBUG - IRC JOIN %s %s\n", channel, key);
#endif
    fdprintf(icsfd, "JOIN %s %s\n", channel, key);
  } else {
#ifdef DEBUG
    fprintf(stderr, "DEBUG - IRC JOIN %s\n", channel);
#endif
    fdprintf(icsfd, "JOIN %s\n", channel);
  }
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Quit(const char *message)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC QUIT %s\n", message);
#endif
  fdprintf(icsfd, "QUIT %s\n", message);
#ifdef DEBUG
  fputs("DEBUG - Closing ICS Output stream\n", stderr);
#endif
  close(icsfd);
  icsfd = -1;
}

TF IRC_IsConnected()
{
  return icsint_active;
}

const char * IRC_GetNick()
{
  return bn2_our_nick;
}

const char * IRC_GetHost()
{
  return bn2_our_host;
}

const char * IRC_HostmaskNick(const char *prefix)
{
  if (prefix!=bn2_internal_buffer)
    Util_Copy(bn2_internal_buffer, prefix, ALLOC_MINIBUFFER);
  if (strchr(bn2_internal_buffer, '!')!=NULL)
    PrefixSeparate();
  return bn2_internal_buffer;
}

const char * IRC_HostmaskUser(const char *prefix)
{
  const char *temp;
  if (prefix!=bn2_internal_buffer)
    Util_Copy(bn2_internal_buffer, prefix, ALLOC_MINIBUFFER);
  if (strchr(bn2_internal_buffer, '!')!=NULL)
    PrefixSeparate();
  temp = strchr(bn2_internal_buffer, 0);
  return temp+1;
}

const char * IRC_HostmaskHost(const char *prefix)
{
  const char *temp;
  if (prefix!=bn2_internal_buffer)
    Util_Copy(bn2_internal_buffer, prefix, ALLOC_MINIBUFFER);
  if (strchr(bn2_internal_buffer, '!')!=NULL)
    PrefixSeparate();
  temp = strchr(bn2_internal_buffer, 0);
  temp = strchr(temp+1, 0);
  return temp+1;
}

void * IRC_DCCChatSend(const char *target)
{
  int sock;
  struct sockaddr_in sin;
  size_t len;
  char buffer[ALLOC_MINIBUFFER];

#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC DCC CHAT %s\n", target);
#endif
  //  fprintf(icsout, "DCCCHAT %s\n", target);
  sock = socket(PF_INET, SOCK_STREAM, 0);
  if (sock<0) {
    fprintf(stderr, "IRC_DCCChatSend: Unable to create socket: %m\n");
    return NULL;
  }
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = htonl(INADDR_ANY);
  sin.sin_port = 0;
  len = 1;
  setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (void *)(&len), sizeof(len));
  if (bind(sock, (struct sockaddr *)(&sin), sizeof(struct sockaddr_in))<0) {
    fprintf(stderr, "IRC_DCCChatSend: Unable to bind socket: %m\n");
    close (sock);
    return NULL;
  }
  len = sizeof(struct sockaddr_in);
  if (getsockname(sock, (struct sockaddr *)(&sin), &len)<0) {
    fprintf(stderr, "IRC_DCCChatSend: Unable to read address: %m\n");
    close (sock);
    return NULL;
  }
  if (listen(sock, 1)<0) {
    fprintf(stderr, "IRC_DCCChatSend: Unable to listen: %m\n");
    close (sock);
    return NULL;
  }
  NewDCC(target, sock);
  snprintf(buffer, ALLOC_MINIBUFFER, "CHAT chat %lu %d", bn2_our_addr, 
           ntohs(sin.sin_port));
  IRC_CTCP(target, "DCC", buffer);
  return (void *)dcc_list;
}

void IRC_DCCChatAccept(void *cookie)
{
  int sock, ioflags;

#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC DCC CHAT ACCEPT %s\n", ((PDCC)cookie)->nick);
#endif
  //  fprintf(icsout, "DCCCHATACCEPT %s\n", ((PDCC)cookie)->cookie);
  sock = socket(PF_INET, SOCK_STREAM, 0);
  if (sock<0) {
    fprintf(stderr, "IRC_DCCChatAccept: Unable to create socket: %m\n");
    return ;
  }
  if (connect(sock, (struct sockaddr *)&(((PDCC)cookie)->addr), 
              ((PDCC)cookie)->addrlen)<0) {
    fprintf(stderr, "IRC_DCCChatAccept: Unable to connect: %m\n");
    close (sock);
    return ;
  }

  //  ioflags=fcntl(sock, F_GETFL, 0);
  //  ioflags|=O_NONBLOCK;
  //  fcntl(sock, F_SETFL, ioflags);

  ((PDCC)cookie)->sock = sock;
  FD_SET(sock, read_fds);
  HandleDCCChatOpened(cookie);
}

void IRC_DCCChatReject(void *cookie)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC DCC CHAT REJECT %s\n", ((PDCC)cookie)->nick);
#endif
  //  fprintf(icsout, "DCCCHATREJECT %s\n", ((PDCC)cookie)->cookie);
  IRC_CTCPReply(((PDCC)cookie)->nick, "ERRMSG", "DCC CHAT chat declined");
  RemDCC((PDCC)cookie);
}

void IRC_DCCChatMessage(void *cookie, const char *message)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC DCC MSG %s %s\n", ((PDCC)cookie)->nick, message);
#endif
  //  fprintf(icsout, "DCCCHATMSG %s %s\n", ((PDCC)cookie)->cookie, message);
  fdprintf(((PDCC)cookie)->sock, "%s\n", message);
  //  FD_SET(fileno(((PDCC)cookie)->dccout), write_fds);
}

void IRC_DCCChatClose(void *cookie)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC DCC CHAT CLOSE%s\n", ((PDCC)cookie)->nick);
#endif
  //  fprintf(icsout, "DCCCHATCLOSE %s\n", ((PDCC)cookie)->cookie);
  RemDCC((PDCC)cookie);
}

const char * IRC_DCCGetNick(void *cookie)
{
  return ((PDCC)cookie)->nick;
}

void IRC_DCCSend(const char *target, const char *fn)
{
#ifdef DEBUG
  fprintf(stderr, "DEBUG - IRC DCC SEND %s %s\n", target, fn);
#endif
  //  fprintf(icsout, "DCCSEND %s %s\n", target, fn);
}

void OnDebug(const char *str)
{
#ifdef DEBUG
  fprintf(stderr, "ICS: %s\n", str);
#endif
}

void OnVersion(const char *str)
{
  if (ICSINFO!=NULL)  // Stale
    free (ICSINFO);
  ICSINFO=strdup(str);
}

void OnDCCChatClose(PDCC dcc)
{
  HandleDCCChatClosed((void *)dcc);
  dcc->stale = TRUE;
}

void OnDCCChatMSG(PDCC dcc, const char *str)
{
  char msg[ALLOC_BUFFER], *temp;
  Util_Copy(msg, str, ALLOC_BUFFER);
  temp = strchr(msg, '\n');
  if (temp != NULL)
    *temp = 0;
  HandleDCCChatMessage((void*)(dcc), msg);
}

void OnConnect(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  bn2_our_addr = strtoul(s1, NULL, 10);
  Util_Copy(bn2_our_host, s2, ALLOC_HOST);
  fdprintf(icsfd, "REGISTER %s %s %s\n", bn2_our_nick, bn2_our_user,
	   bn2_our_name);
  //  FD_SET(fileno(icsout), write_fds);
}

void OnNumeric(const char *str)
{
  char *temp;
  long int num;
  num = strtol(str, &temp, 10);
  HandleNumeric(num, temp);
}

void OnRegistered(const char *str)
{
  ConnectionReady();
}

void OnUnknown(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  fprintf(stderr, "Unknown event from %s : %s\n", s1, s2);
}

void OnError(const char *str)
{
  fprintf(stderr, "Event error: %s\n", str);
}

void OnNotice(const char *str)
{
  const char **ptr = (const char **)malloc(sizeof(char *)*3);
  Util_Explode(str, 32, ptr, 3);
  HandleNotice(ptr[0], ptr[1], ptr[2]);
  free(ptr);
}

void OnDCC(const char *str) {
  if (strncasecmp(str, "CHAT chat ", 10)==0 ||
      strncasecmp(str, "CHAT clear ", 11)==0) {
    unsigned long addr;
    unsigned short port;
    const char *temp;
    PrefixSeparate();
    temp = strchr(str, 32);
    temp = strchr(temp+1, 32);
    addr = strtoul(temp+1, NULL, 10);
    temp = strchr(temp+1, 32);
    port = strtoul(temp+1, NULL, 10);
    if (!addr||!port) {
      char pr[6];
      Util_Copy(pr, strchr(str, 32)+1, 6);
      if (pr[4]==32)
	pr[4] = 0;
      IRC_CTCPReply(bn2_internal_buffer, "ERRMSG", 
		    Util_Format("DCC CHAT %s missing or incorrect address", 
				pr, NULL, NULL));
      return ;
    }
    NewDCC(bn2_internal_buffer, -1);
    dcc_list->addr.sin_family = AF_INET;
    dcc_list->addr.sin_addr.s_addr = htonl(addr);
    dcc_list->addr.sin_port = htons(port);
#ifdef DEBUG
    fprintf(stderr, "DCC CHAT to %s:%u\n", inet_ntoa(dcc_list->addr.sin_addr),
	    port);
#endif
    *strchr(bn2_internal_buffer, 0) = '!';
    *strchr(bn2_internal_buffer, 0) = '@';
    HandleDCCChatRequestPfx((void *)dcc_list, bn2_internal_buffer);
  }
}

void OnCTCP(const char *str)
{
  const char *s1, *s2, *temp;
  s2 = str;
  Util_SepTo(bn2_internal_buffer, ALLOC_MINIBUFFER, &s2, 32);
  s1 = Util_Separate(&s2, 32);
  if (strncasecmp(s2, "DCC ", 4)==0 && strcasecmp(s1, bn2_our_nick)==0) {
    OnDCC(s2+4);
    return ;
  }
  if ((temp = HandleCTCP(bn2_internal_buffer, s1, s2))!=NULL) {
    PrefixSeparate();
    IRC_CTCPReply(bn2_internal_buffer, s1, temp);
  }
}

void OnCTCPReply(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  HandleCTCPReply(s1, s2);
}

void OnName(const char *str)
{
  // Under the circumstances, it seems best to send one name at a time
  const char *s1, *s2;

  s2 = str;
  s1 = Util_Separate(&s2, 32);
  HandleName(s1, s2);
}

void OnInvite(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  HandleInvite(s1, s2);
}

void OnKick(const char *str)
{
  const char *temp[4];
  Util_Explode(str, 32, temp, 4);
  HandleKick(IRC_HostmaskNick(temp[0]), temp[1], temp[2], temp[3]);
}

void OnMSG(const char *str)
{
  const char *temp;
  temp = str;
  Util_SepTo(bn2_internal_buffer, ALLOC_MINIBUFFER, &temp, 32);
  HandleMSG(bn2_internal_buffer, temp);
}

void OnPublic(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  Util_SepTo(bn2_internal_buffer, ALLOC_MINIBUFFER, &s2, 32);
  s1 = Util_Separate(&s2, 32);
  HandlePublic(bn2_internal_buffer, s1, s2);
}

void OnAction(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  Util_SepTo(bn2_internal_buffer, ALLOC_MINIBUFFER, &s2, 32);
  s1 = Util_Separate(&s2, 32);
  HandleCTCP(bn2_internal_buffer, s1, Util_Format("%s %s", "ACTION", s2, NULL));
}

void OnJoin(const char *str)
{
  int c;
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  c = strchr(s1, '!') - s1;
  HandleJoin(s1, s2);
}

void OnJoined(const char *str)
{
  HandleJoined(str);
}

void OnNick(const char *str)
{
  int c;
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  c = strchr(s1, '!') - s1;
  if (strncasecmp(s1, bn2_our_nick, c)==0) {
    HandleOurNickChanged(s2);
    Util_Copy(bn2_our_nick, s2, ALLOC_NICK);
  } else
    HandleNickChange(s1, s2);
}

void OnPart(const char *str)
{
  char nick[ALLOC_NICK];
  const char *s1, *s2;
  s2 = str;
  Util_SepTo(nick, ALLOC_NICK, &s2, 32);
  s1 = Util_Separate(&s2, 32);
  if (s1 == NULL)
    HandlePart(nick, s2, NULL);
  else
    HandlePart(nick, s1, s2);
}

void OnQuit(const char *str)
{
  const char *nick, *msg;
  msg = str;
  nick = Util_Separate(&msg, 32);
  if (strcasecmp(nick, bn2_our_nick)==0)
    return ;
  HandleQuit(nick, msg);
}

void OnDisconnect(const char *message)
{
  HandleDisconnect(message);
  close(icsfd);
  icsfd = -1;
  while (dcc_list != NULL)
    RemDCC(dcc_list);
  FD_ZERO(read_fds);
  //  FD_ZERO(write_fds);
}

struct strtofunc_map {
  const char *str;
  void (*fun)(const char *);
};

const struct strtofunc_map EVENT_LIST[] = {
  { "DEBUG", &OnDebug },
  { "VERSION", &OnVersion },
  { "CONNECT", &OnConnect },
  { "NUMERIC", &OnNumeric },
  { "REGISTERED", &OnRegistered },
  { "UNKNOWN", &OnUnknown },
  { "ERROR", &OnError },
  { "NOTICE", &OnNotice },
  { "CTCP", &OnCTCP },
  { "CTCPREPLY", &OnCTCPReply },
  { "NAME", &OnName },
  { "INVITE", &OnInvite },
  { "KICK", &OnKick },
  { "MSG", &OnMSG },
  { "PUBLIC", &OnPublic },
  { "ACTION", &OnAction },
  { "JOIN", &OnJoin },
  { "JOINED", &OnJoined },
  { "NICK", &OnNick },
  { "PART", &OnPart },
  { "QUIT", &OnQuit },
  { "DISCONNECT", &OnDisconnect },
  { NULL, NULL }
};

void StartMessage()
{
  fd_set reads;
  char *input = NULL;
  size_t len = 0;
  const char *cmd, *data;
  int c;
  PDCC dcc;
  while (1) {
#ifdef DEBUG
    fputs("DEBUG - Top of StartMessage loop.\n", stderr);
#endif
    memcpy(&reads, read_fds, sizeof(fd_set));
    //    memcpy(&writes, write_fds, sizeof(fd_set));
    select(FD_SETSIZE, &reads, NULL, NULL, NULL);

    for (dcc = dcc_list; dcc != NULL; dcc = dcc->next)
      if (dcc->sock != -1 && FD_ISSET(dcc->sock, &reads)) {
	if (dcc->listen)
	  AcceptDCC(dcc);
	else {
	  int r = fdgetline(&input, &len, dcc->sock);
	  if (r<1)
	    OnDCCChatClose(dcc);
	  else {
	    *strchr(input, '\n')=0;
	    OnDCCChatMSG(dcc, input);
	  }
	}
      }
    DoDCCCleanup();

    if (FD_ISSET(icsfd, &reads)) {
      if (fdgetline(&input, &len, icsfd)<0) {
	fprintf(stderr, "Error reading socket: %m\n");
	exit (255);
      }
      *(strchr(input, '\n'))=0;
#ifdef DEBUG
      fprintf(stderr, "DEBUG - Received from ICS: %s\n", input);
#endif
      data = (const char *)(input);
      cmd = Util_Separate(&data, 32);
      if (cmd==NULL) {
	cmd = data;
	data = NULL;
      }
      for (c=0;EVENT_LIST[c].str!=NULL;c++)
	if (strcasecmp(cmd, EVENT_LIST[c].str)==0) {
	  (*(EVENT_LIST[c].fun))(data);
	  break;
	}
    }
  }
}

void IRC_Connect(const char *server, const int port, const char *nick,
		 const char *user, const char *s_name)
{
  int sock[2], res, ioflags;
  char pn[6];
  const char *fn;
#ifdef USE_DEBUG_ICS
  struct sockaddr_un *sunaddr;
  struct sockaddr *saddr;
  socklen_t *slen;
#endif

#ifdef DEBUG
  fputs("DEBUG - IRC_Connect called\n", stderr);
#endif

  read_fds = (fd_set *)malloc(sizeof(fd_set));
  //  write_fds = (fd_set *)malloc(sizeof(fd_set));
  FD_ZERO(read_fds);
  //  FD_ZERO(write_fds);

  Util_Copy(bn2_our_nick, nick, ALLOC_NICK);
  Util_Copy(bn2_our_user, user, ALLOC_USER);
  Util_Copy(bn2_our_name, s_name, ALLOC_NAME);
#ifndef USE_DEBUG_ICS
  if (*options.interface_fn)
    fn = (const char *)options.interface_fn;
  else
    fn = "lachesis-ics";

  snprintf(pn, 6, "%d", port);

  if (socketpair(AF_LOCAL, SOCK_STREAM, 0, sock)<0) {
    fprintf(stderr, "Unable to allocate socket pair: %m\n");
    exit (1);
  }

  if ((res=fork())==0) {
    dup2(sock[1], STDIN_FILENO);
    dup2(sock[1], STDOUT_FILENO);
    execl(fn, fn, server, pn, NULL);
    fprintf(stderr, "ICS exec failure: %m\n");
    _exit(EXIT_FAILURE);
  }

  if (res<0) {
    fprintf(stderr, "Fork failure: %m\n");
    exit (1);
  }
#else
  sock[1] = socket(PF_LOCAL, SOCK_STREAM, 0);
  if (sock[1]<0) {
    fprintf(stderr, "Failed to create socket: %m\n");
    exit (1);
  }
  sunaddr = (struct sockaddr_un*)malloc(sizeof(short int)+strlen("/tmp/lachesis-debug")+1);
  sunaddr->sun_family=AF_LOCAL;
  strcpy(sunaddr->sun_path, "/tmp/lachesis-debug");
  if (bind(sock[1], (struct sockaddr *)sunaddr, SUN_LEN(sunaddr))<0) {
    fprintf(stderr, "Failed to bind socket: %m\n");
    exit (1);
  }
  chmod("/tmp/lachesis-debug", S_IRWXU);
  listen(sock[1], 1);
  sock[0] = accept(sock[1], saddr, slen);
  if (sock[0]<0) {
    fprintf(stderr, "Error from accept: %m\n");
    exit (1);
  }
  free (sunaddr);
  close (sock[1]);
  unlink("/tmp/lachesis-debug");
#endif

  //  ioflags=fcntl(sock[0], F_GETFL, 0);
  //  ioflags|=O_NONBLOCK;
  //  fcntl(sock[0], F_SETFL, ioflags);

  FD_SET(sock[0], read_fds);
  icsfd = sock[0];

  StartMessage();
}

void LowIrcCleanup()
{
#ifdef DEBUG
  fputs("DEBUG - LowIrcCleanup called\n", stderr);
#endif
  if (read_fds!=NULL) {
    free (read_fds);
    read_fds = NULL;
  }
  //  if (write_fds!=NULL) {
  //    free (write_fds);
  //    write_fds = NULL;
  //  }
  if (icsfd!=-1) {
    close(icsfd);
    icsfd = -1;
  }
  while (dcc_list != NULL)
    RemDCC(dcc_list);
  if (ICSINFO!=NULL) {
    free (ICSINFO);
    ICSINFO=NULL;
  }
}
