/*
 * Lachesis, an IRCRPG combat engine - Authentication command handler
 * Copyright 2002-2004 M. Dennis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <stdio.h>
#include "root.h"
#include "authcmd.h"
#include "../auth.h"
#include "../ircint.h"
#include "../utils.h"
#include "../loaddata.h"
#include "../config.h"

static char* id="@(#) $Id: authcmd.cc,v 1.1.1.1 2004/01/16 07:28:15 lachesis Exp $";

const char *CMD_AUTHCMD_USERHELP[] =
  { "LOGIN", "AUTH LOGIN authenticates users.",
    "LOGOUT", "AUTH LOGOUT will log out of an account.",
    "PASSWD", NULL,
    "PASSWORD", "AUTH PASSWORD <old> <new> will change your password. You must provide the correct current password. You may also use AUTH PASSWD <old> <new>.",
    NULL, "The authentication system provides access to account settings, as well as logging into and out of an account.",
  };

const char *CMD_AUTHCMD_SETHELP[] =
  { "SECURITY", "AUTH SET SECURITY [ON|OFF] - Sets or queries the high security option.\nBy default, hostmasks aren't completely trusted; setting this option to OFF will allow any matching hostmask when users aren't seen in channels.",
    "USEMSG", "AUTH SET USEMSG [ON|OFF] - Sets or queries the MSG option.\nBy default, when no DCC CHAT connection is present, information is sent to the user using notices. Setting this option to ON will send using MSG instead.",
#ifndef RPGSERV
    "SOUNDS", "AUTH SET SOUNDS [ON|OFF] - Sets or queries the sound option.\nThis currently does not affect anything.",
#endif
    NULL, "AUTH SET maintains account settings. The following options are available:\n  SECURITY\n  USEMSG\n  SOUNDS",
  };

const char *CMD_AUTHCMD_NONEHELP = "Available commands:\n  AUTH LOGIN <acct> <pass>\n    Log into a user account.\n  HELP\n    Display this message.";

#ifndef RPGSERV
extern Cmd_PMod cmd_modules;

void Cmd_authcmd_Init()
{
  Cmd_Alloc(&Cmd_Auth_Root, "auth", CMD_AUTH_NONE);
}
#endif

void Cmd_Auth_Root(void *ticket, int auth, const char *str)
{
  const char *data, *cmd;

  if (str==NULL) {
    Auth_CmdResult(ticket, auth, 
		   "Usage: auth {help|login|logout|password|set}");
    return ;
  }
  data = str;
  cmd = Util_Separate(&data, 32);
  if (cmd==NULL) {
    cmd = data;
    data = NULL;
  }
  if (strcasecmp(cmd, "LOGIN")==0) {
    Cmd_Auth_HandleLogin(ticket, auth, data);
    return ;
  }
  if (strcasecmp(cmd, "HELP")==0) {
    Cmd_Auth_HandleHelp(ticket, auth, data);
    return ;
  }
  if (auth<CMD_AUTH_USER) {
    Auth_CmdResult(ticket, auth, 
		   "You must log in first. AUTH LOGIN <acct> <pass>");
    return ;
  }
  if (strcasecmp(cmd, "LOGOUT")==0) {
    Auth_CmdResult(ticket, auth, "Logging out...");
    Auth_Rem(ticket, auth);
  }
  if (strcasecmp(cmd, "PASSWORD")==0 ||
      strcasecmp(cmd, "PASSWD")==0) {
    Cmd_Auth_HandlePasswd(ticket, auth, data);
    return ;
  }
  if (strcasecmp(cmd, "SET")==0) {
    Cmd_Auth_HandleSet(ticket, auth, data);
    return;
  }
  if (strcasecmp(cmd, "DCC")==0) {
    void *cookie;
    Auth_PUser temp;
    if (auth==CMD_AUTH_MASTER)
      temp = Auth_TUser::GetMaster();
    else
      temp = (Auth_PUser)ticket;
    cookie = IRC_DCCChatSend(temp->GetNick());
    if (cookie!=NULL) {
      void *oldcookie = temp->RegisterDCC(cookie);
      if (oldcookie!=NULL)
	IRC_DCCChatClose(oldcookie);
    }
    return ;
  }
  Auth_CmdResult(ticket, auth, 
		 Util_FormatD("Auth: %s not recognized as a command.", cmd,
			      NULL, NULL)
		 );
}

void Cmd_Auth_HandleLogin(void *ticket, int auth, const char *str) {
  void *temp;
  temp = ticket;
  if (auth!=CMD_AUTH_NONE) {
    Auth_CmdResult(ticket, auth, "You're already logged in.");
    return ;
  }
  char acct[ALLOC_ACCT], pass[ALLOC_PASS];
  const char *ts;

  if (str==NULL || strchr(str, 32)==NULL)
    Auth_CmdResult(ticket, auth, "Usage: auth login <acct> <pass>");
  ts = str;
  Util_SepTo(acct, ALLOC_ACCT, &ts, 32);
  if (strchr(ts, 32))
    Util_SepTo(pass, ALLOC_PASS, &ts, 32);
  else
    Util_Copy(pass, ts, ALLOC_PASS);
  if (Auth_Add(&temp, acct, pass))
    Auth_CmdResult(temp, CMD_AUTH_USER, "Login successful.");
  else
    Auth_CmdResult(ticket, CMD_AUTH_NONE, "Login failed.");
}

void Cmd_Auth_HandleHelp(void *ticket, int auth, const char *str)
{
  const char *out, *buf;
  if (auth<CMD_AUTH_USER) {
    buf = CMD_AUTHCMD_NONEHELP;
    while (out = Util_Separate(&buf, 10))
      Auth_CmdResult(ticket, auth, out);
    Auth_CmdResult(ticket, auth, buf);
    return ;
  }
  int c;
  const char *data, *cmd;
  if (str==NULL)
    str = "garbage-string";
  data = str;
  cmd = Util_Separate(&data, 32);
  if (!cmd) {
    cmd = data;
    data = NULL;
  }
  if (strcasecmp(cmd, "SET")==0) {
    if (data) {
      cmd = Util_Separate(&data, 32);
      if (!cmd) {
	cmd = data;
	data = NULL;
      }
    }
    for (c=0;CMD_AUTHCMD_SETHELP[c];c+=2) {
      if (strcasecmp(cmd, CMD_AUTHCMD_SETHELP[c])==0) {
	while (!(buf = CMD_AUTHCMD_SETHELP[c+1]))
	  c+=2;
	while (out = Util_Separate(&buf, 10))
	  Auth_CmdResult(ticket, auth, out);
	Auth_CmdResult(ticket, auth, buf);
	return ;
      }
    }
    buf = CMD_AUTHCMD_SETHELP[c+1];
    while (out = Util_Separate(&buf, 10))
      Auth_CmdResult(ticket, auth, out);
    Auth_CmdResult(ticket, auth, buf);
    return ;
  }
  for (c=0;CMD_AUTHCMD_USERHELP[c];c+=2) {
    if (strcasecmp(cmd, CMD_AUTHCMD_USERHELP[c])==0) {
      while (!(buf = CMD_AUTHCMD_USERHELP[c+1]))
	c+=2;
      while (out = Util_Separate(&buf, 10))
	Auth_CmdResult(ticket, auth, out);
      Auth_CmdResult(ticket, auth, buf);
      return ;
    }
  }
  buf = CMD_AUTHCMD_USERHELP[c+1];
  while (out = Util_Separate(&buf, 10))
    Auth_CmdResult(ticket, auth, out);
  Auth_CmdResult(ticket, auth, buf);
}

void Cmd_Auth_HandlePasswd(void *ticket, int auth, const char *str)
{
  //  Auth_CmdResult(ticket, auth, "Sorry, this isn't implemented yet.");
  char oldpass[ALLOC_PASS], newpass[ALLOC_PASS];
  const char *ptr = str;
  Auth_PUser tuser;
  Util_SepTo(oldpass, ALLOC_PASS, &ptr, 32);
  if (ptr==NULL) {
    Auth_CmdResult(ticket, auth, "Syntax: AUTH PASSWORD <old> <new>");
    return;
  }
  Util_SepTo(newpass, ALLOC_PASS, &ptr, 32);
  if (auth==CMD_AUTH_MASTER)
    tuser = Auth_TUser::GetMaster();
  else
    tuser = (Auth_PUser)ticket;
  if (strcmp(tuser->pass, oldpass)==0) {
    LD_SetPassword(tuser->acct, newpass);
    strncpy(tuser->pass, newpass, ALLOC_PASS-1);
    tuser->pass[ALLOC_PASS-1] = '\0';
    Auth_CmdResult(ticket, auth, "Password changed.");
  } else
    Auth_CmdResult(ticket, auth, "Incorrect password.");
}

void Cmd_Auth_HandleSet(void *ticket, int auth, const char *str)
{
  if (!str||!*str) {
    Auth_CmdResult(ticket, auth, "Usage: AUTH SET SECURITY|USEMSG|SOUNDS [ON|OFF]");
    return ;
  }
  const char *cmd, *data;
  data = str;
  cmd = Util_Separate(&data, 32);
  if (!cmd) {
    cmd = data;
    data = NULL;
  }
  Auth_PUser tuser;
  if (auth==CMD_AUTH_MASTER)
    tuser = Auth_TUser::GetMaster();
  else
    tuser = (Auth_PUser)(ticket);
  if(strcasecmp(cmd, "SECURITY")==0) {
    if (!data||!*data) {
      Auth_CmdResult(ticket, auth, Util_Format("Security option is %s.",
					       (tuser->security)?"ON":"OFF",
					       NULL, NULL)
		     );
      return ;
    }
    if (strcasecmp(data, "ON")==0||((cmd=Util_Separate(&data, 32))&&
				    strcasecmp(cmd, "ON")==0)) {
      tuser->security = TRUE;
      LD_SetSecurity(tuser->GetAcct(), TRUE);
      Auth_CmdResult(ticket, auth, "Security option set to ON.");
      return ;
    }
    if ((cmd&&strcasecmp(cmd, "OFF")==0)||(data&&strcasecmp(data, "OFF")==0)) {
      tuser->security = FALSE;
      LD_SetSecurity(tuser->GetAcct(), FALSE);
      Auth_CmdResult(ticket, auth, "Security option set to OFF.");
      return ;
    }
    Auth_CmdResult(ticket, auth, Util_FormatD("Unrecognized setting: %s", 
					      (cmd)?cmd:data, NULL, NULL)
		   );
    return ;
  }
  if(strcasecmp(cmd, "USEMSG")==0) {
    if (!data||!*data) {
      Auth_CmdResult(ticket, auth, Util_Format("UseMSG option is %s.",
					       (tuser->usemsg)?"ON":"OFF",
					       NULL, NULL)
		     );
      return ;
    }
    if (strcasecmp(data, "ON")==0||((cmd=Util_Separate(&data, 32))&&
				    strcasecmp(cmd, "ON")==0)) {
      tuser->usemsg = TRUE;
      LD_SetUseMSG(tuser->GetAcct(), TRUE);
      Auth_CmdResult(ticket, auth, "UseMSG option set to ON.");
      return ;
    }
    if ((cmd&&strcasecmp(cmd, "OFF")==0)||(data&&strcasecmp(data, "OFF")==0)) {
      tuser->usemsg = FALSE;
      LD_SetUseMSG(tuser->GetAcct(), FALSE);
      Auth_CmdResult(ticket, auth, "UseMSG option set to OFF.");
      return ;
    }
    Auth_CmdResult(ticket, auth, Util_FormatD("Unrecognized setting: %s", 
					      (cmd)?cmd:data, NULL, NULL)
		   );
    return ;
  }
#ifndef RPGSERV
  if(strcasecmp(cmd, "SOUNDS")==0) {
    if (!data||!strlen(data)) {
      Auth_CmdResult(ticket, auth, Util_Format("Sounds option is %s.",
					       (tuser->sounds)?"ON":"OFF",
					       NULL, NULL)
		     );
      return ;
    }
    if (strcasecmp(data, "ON")==0||((cmd=Util_Separate(&data, 32))&&
				    strcasecmp(cmd, "ON")==0)) {
      tuser->sounds = TRUE;
      LD_SetSounds(tuser->GetAcct(), TRUE);
      Auth_CmdResult(ticket, auth, "Sounds option set to ON.");
      return ;
    }
    if ((cmd&&strcasecmp(cmd, "OFF")==0)||(data&&strcasecmp(data, "OFF")==0)) {
      tuser->sounds = FALSE;
      LD_SetSounds(tuser->GetAcct(), FALSE);
      Auth_CmdResult(ticket, auth, "Sounds option set to OFF.");
      return ;
    }
    Auth_CmdResult(ticket, auth, Util_FormatD("Unrecognized setting: %s", 
					      (cmd)?cmd:data, NULL, NULL)
		   );
    return ;
  }
#endif
  Auth_CmdResult(ticket, auth, 
		 Util_FormatD("Unrecognized SET option: %s", cmd, NULL, NULL)
		 );
}

